<?php
namespace Jet_Popup\Render;

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

class Elementor_Content_Render extends Base_Render {

	/**
	 * [$name description]
	 * @var string
	 */
	protected $name = 'elementor-template-render';

	/**
	 * [$depended_scripts description]
	 * @var array
	 */
	public $depended_scripts = [];

	/**
	 * @var array
	 */
	public $elementor_widgets = [];

	/**
	 * @var bool
	 */
	public $post_css_exist = false;

	/**
	 * [init description]
	 * @return [type] [description]
	 */
	public function init() {}

	/**
	 * [get_name description]
	 * @return [type] [description]
	 */
	public function get_name() {
		return $this->name;
	}

	/**
	 * @return array
	 */
	public function get_depended_scripts() {
		return array_unique( $this->depended_scripts );
	}

	/**
	 * @return array
	 */
	public function get_elementor_widgets() {
		return array_unique( $this->elementor_widgets );
	}

	/**
	 * [render description]
	 * @return [type] [description]
	 */
	public function render() {

		if ( ! \Jet_Popup_Utils::has_elementor() ) {
			echo __( 'Elementor not installed', 'jet-popup' );

			return false;
		}

		if ( $this->is_editor() ) {
			echo __( 'Elementor editor content is not available in the Blocks Editor', 'jet-popup' );

			return false;
		}

		$popup_id = $this->get( 'popup_id', false );
		$with_css = $this->get( 'with_css', false );

		if ( ! $popup_id ) {
			return false;
		}

		$content = \Elementor\Plugin::instance()->frontend->get_builder_content_for_display( $popup_id, $with_css );

		echo do_shortcode( $content );

    }

	/**
	 * @return array
	 */
	public function get_render_data() {
		$popup_id = $this->get( 'popup_id', false );
		$popup_id = apply_filters( 'jet-popup/popup-generator/before-define-popup-assets/popup-id', $popup_id, $this->get_settings() );

		$template_scripts = [];
		$template_styles = [];
		$font_url_list = $this->get_elementor_template_font_url_list( $popup_id );

		if ( $font_url_list ) {
			$template_styles = wp_parse_args( $template_styles, $font_url_list );
		}

		\Elementor\Plugin::instance()->frontend->register_scripts();

		do_action( 'jet_plugins/frontend/register_scripts' );

		$scripts_data = $this->get_elementor_template_scripts( $popup_id );

		if ( ! empty( $scripts_data['depends'] )) {
			foreach ( $scripts_data['depends'] as $script ) {
				$script_handler = $this->get_script_uri_by_handler( $script );

				if ( $script_handler ) {
					$template_scripts[ $script ] = $script_handler;
				}
			}
		}

		$render_data = [
			'content' => $this->get_content(),
			'scripts' => $template_scripts,
			'styles'  => $template_styles,
		];

		return apply_filters( 'jet-popup/elementor-render/render-data', $render_data, $popup_id, $scripts_data['widgets'] );
	}

	/**
	 * @param $template_id
	 *
	 * @return false|string
	 */
	public function get_elementor_template_font_url_list( $popup_id ) {

		$transient_key = md5( sprintf( 'jet_popup_content_render_data_styles_%s', $popup_id ) );
		$data_cache = get_transient( $transient_key );
		$use_cache = apply_filters( 'jet-popup/elementor-render/render-data/use-cache', true );

		if ( ! empty( $data_cache ) && $use_cache ) {
			return $data_cache;
		}

		$post_css = new \Elementor\Core\Files\CSS\Post( $popup_id );
		$post_meta = $post_css->get_meta();
		$this->post_css_exist = ! empty( $post_meta['status'] ) ? true : false;
		$urls_list = [];

		if ( ! isset( $post_meta['fonts'] ) ) {
			return false;
		}

		$fonts = $post_meta['fonts'];

		if ( empty( $fonts ) ) {
			return false;
		}

		$fonts = array_unique( $fonts );

		$google_fonts = [
			'google' => [],
			'early' => [],
		];

		foreach ( $fonts as $key => $font ) {
			$font_type = \Elementor\Fonts::get_font_type( $font );

			switch ( $font_type ) {
				case \Elementor\Fonts::GOOGLE:
					$google_fonts['google'][] = $font;
					break;

				case \Elementor\Fonts::EARLYACCESS:
					$google_fonts['early'][] = $font;
					break;
			}
		}

		if ( ! empty( $google_fonts['google'] ) ) {
			foreach ( $google_fonts['google'] as &$font ) {
				$font = str_replace( ' ', '+', $font ) . ':100,100italic,200,200italic,300,300italic,400,400italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic';
			}

			$fonts_url = sprintf( 'https://fonts.googleapis.com/css?family=%s', implode( rawurlencode( '|' ), $google_fonts['google'] ) );

			$subsets = [
				'ru_RU' => 'cyrillic',
				'bg_BG' => 'cyrillic',
				'he_IL' => 'hebrew',
				'el'    => 'greek',
				'vi'    => 'vietnamese',
				'uk'    => 'cyrillic',
				'cs_CZ' => 'latin-ext',
				'ro_RO' => 'latin-ext',
				'pl_PL' => 'latin-ext',
				'hr_HR' => 'latin-ext',
				'hu_HU' => 'latin-ext',
				'sk_SK' => 'latin-ext',
				'tr_TR' => 'latin-ext',
				'lt_LT' => 'latin-ext',
			];

			$locale = get_locale();

			if ( isset( $subsets[ $locale ] ) ) {
				$fonts_url .= '&subset=' . $subsets[ $locale ];
			}

			$urls_list[ "jet-popup-google-fonts-{$popup_id}" ] = $fonts_url;

		}

		if ( ! empty( $google_fonts['early'] ) ) {
			foreach ( $google_fonts['early'] as $current_font ) {
				$font_url = sprintf( 'https://fonts.googleapis.com/earlyaccess/%s.css', strtolower( str_replace( ' ', '', $current_font ) ) );
				$urls_list[ "jet-popup-google-earlyaccess-{$popup_id}" ] = $font_url;
			}
		}

		if ( $this->post_css_exist ) {
			set_transient( $transient_key, $urls_list, 12 * HOUR_IN_SECONDS );
		}

		return $urls_list;
	}

	/**
	 * [get_elementor_template_scripts_url description]
	 * @param  [type] $template_id [description]
	 * @return [type]              [description]
	 */
	public function get_elementor_template_scripts( $popup_id = false ) {

		if ( ! $popup_id ) {
			return [
				'depends' => [],
				'widgets' => [],
			];
		}

		$transient_key = md5( sprintf( 'jet_popup_content_render_data_scripts_%s', $popup_id ) );
		$data_cache = get_transient( $transient_key );
		$use_cache = apply_filters( 'jet-popup/elementor-render/render-data/use-cache', true );

		if ( ! empty( $data_cache ) && $use_cache ) {
			return $data_cache;
		}

		$document = \Elementor\Plugin::$instance->documents->get( $popup_id );

		if ( ! $document ) {
			return [
				'depends' => [],
				'widgets' => [],
			];
		}

		$elements_data = $document->get_elements_raw_data();

		if ( empty( $elements_data ) ) {
			return [
				'depends' => [],
				'widgets' => [],
			];
		}

		$this->find_widgets_script_handlers( $elements_data );

		$scripts_data = [
			'depends' => array_unique( $this->depended_scripts ),
			'widgets' => array_unique( $this->elementor_widgets ),
		];

		set_transient( $transient_key, $scripts_data, 12 * HOUR_IN_SECONDS );

		return $scripts_data;
	}

	/**
	 * [find_widgets_script_handlers description]
	 * @param  [type] $elements_data [description]
	 * @return [type]                [description]
	 */
	public function find_widgets_script_handlers( $elements_data ) {

		foreach ( $elements_data as $element_data ) {

			if ( 'widget' === $element_data['elType'] ) {
				$widget                    = \Elementor\Plugin::$instance->elements_manager->create_element_instance( $element_data );
				$widget_script_depends     = $widget->get_script_depends();
				$this->elementor_widgets[] = $widget->get_name();

				if ( ! empty( $widget_script_depends ) ) {
					foreach ( $widget_script_depends as $key => $script_handler ) {
						$this->depended_scripts[] = $script_handler;
					}
				}
			} else {
				$element = \Elementor\Plugin::$instance->elements_manager->create_element_instance( $element_data );
				$childrens = $element->get_children();

				foreach ( $childrens as $key => $children ) {
					$children_data[ $key ] = $children->get_raw_data();
					$this->find_widgets_script_handlers( $children_data );
				}
			}
		}
	}

	/**
	 * [get_script_uri_by_handler description]
	 * @param  [type] $handler [description]
	 * @return [type]          [description]
	 */
	public function get_script_uri_by_handler( $handler ) {
		global $wp_scripts;

		if ( isset( $wp_scripts->registered[ $handler ] ) ) {
			$src = $wp_scripts->registered[ $handler ]->src;

			if ( ! preg_match( '|^(https?:)?//|', $src ) && ! ( $wp_scripts->content_url && 0 === strpos( $src, $wp_scripts->content_url ) ) ) {
				$src = $wp_scripts->base_url . $src;
			}

			return $src;
		}

		return false;
	}


}
